; Wind Chime Player
; Y2020 Silicon Chip

	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F1459
	#include p16f1459.inc

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _FOSC_INTOSC & _WDTE_OFF & _PWRTE_ON & _MCLRE_OFF & _CP_OFF & _BOREN_OFF & _CLKOUTEN_OFF &_IESO_OFF & _FCMEN_OFF

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _WRT_OFF & _CPUDIV_NOCLKDIV & _USBLSCLK_24MHz & _PLLMULT_4x & _PLLEN_DISABLED & _STVREN_ON & _BORV_HI & _LPBOR_OFF & _LVP_OFF 


; Define variables at memory locations

; Bank 0 RAM

VALUE_1		equ	H'20'	; delay values 1,2,3
VALUE_2		equ	H'21'
VALUE_3		equ	H'22'
STORE3			equ	H'23'	; A/D store for aquisition
TEMP			equ	H'24'
FSR0L_TMP		equ	H'25'	; FSR temporary store	
COUNT			equ	H'26'	; solenoid on period, JP1 out at RA0
DUTY			equ H'27'	; solenoid drive duty cycle, JP1 in at RA0
NATURAL		equ	H'28'	; natural variation flags with randomisation
SOLENOID		equ	H'29'	; solenoid number (1-12)
TEN_MS		equ	H'2A'	; 10ms period via timer 1
STARTED_REC	equ	H'2B'	; started recording flag
TIMER_MS		equ	H'2C'	; timer for pause between solenoids ms byte
TIMER_LS		equ	H'2D'	; timer ls byte
TEMP_RAMMS	equ	H'2E'	; temporary RAM ms byte
TEMP_RAMLS	equ	H'2F'	; temporaty RAM ls byte
PORTA_STO		equ	H'30'	; port store
PORTB_STO		equ	H'31'	; port store
PORTC_STO	equ	H'32'	; port store
TEST_COUNT	equ	H'33'	; startup delay
FLASH_LED		equ	H'34'	; LED flash counter
SELECTED1		equ	H'35'	; solenoids selected 1-8 in bits 0-7
SELECTED2		equ	H'36'	; solenoids selected 9-12 in bits 0-3
SWITCH			equ	H'37'	; S13 pressed flag
PLAYING		equ	H'38'	; play flag bit 0
SW_HOLD		equ	H'39'	; switch hold counter
COUNT_VAL		equ	H'3A'	; storage of COUNT
COUNT_STO	equ	H'3B'	; count value working value
WORKING		equ	H'3C'	; working value
	
; math files/ random generator 
BARGB0		equ	H'59'	; random number generator
BARGB1		equ	H'5A'	; random number generator
BARGB2		equ	H'5B'	; random number generator
AARGB0		equ	H'5C'	; random number gen
AARGB1		equ	H'5D'	; random number gen
AARGB2		equ	H'5E'	; random number gen
AARGB3		equ	H'5F'	; random number gen
AARGB4		equ	H'60'	; random number gen
AARGB5		equ	H'61'	; random number gen
AARGB6		equ	H'62'	; random number gen
RANDB0		equ	H'63'	; random number seed
RANDB1		equ	H'64'	; random number seed
RANDB2		equ	H'65'	; random number seed
RANDB3		equ	H'66'	; random number seed
TEMPB0		equ	H'67'	; random gen temp files
TEMPB1		equ	H'68'	; random gen temp files
TEMPB2		equ	H'69'	; random gen temp files
TEMPB3		equ	H'6A'	; random gen temp files
LOOPCOUNT	equ	H'6B'	; loop counter in random gen
REMB0			equ	H'6C'	; remainder
TEMP1			equ	H'6D'	; temporary
RNDM1			equ	H'6E'	; random value
RNDM2			equ	H'6F'	; random value

; all banks ram

TEMP_VAL		equ	H'70'	; read data ms byte value
ADDRL			equ	H'71'	; lower byte flash address
ADDRH			equ	H'72'	; upper byte address boundary
READADDMS	equ	H'73'	; ms Byte of Program Address to read
READADDLS	equ	H'74'	; ls Byte of Program Address to read

; initial values
	org	H'1000'	; start address of where data memory is stored for solenoid drive (RAM is from H2050 to H2070) for 32 bits (H1000 to H100C are used for the 12 sets of data)
; 13 spaces reserved for solenoid 1-12 and randomness
	DA			H'3FFF'		 ; COUNT and DUTY Solenoid 1
	DA			H'3FFF'		 ; COUNT and DUTY Solenoid 2
	DA			H'3FFF'		 ; COUNT and DUTY Solenoid 3
	DA			H'3FFF'		 ; COUNT and DUTY Solenoid 4
	DA			H'3FFF'		 ; COUNT and DUTY Solenoid 5
	DA			H'3FFF'		 ; COUNT and DUTY Solenoid 6
	DA			H'3FFF'		 ; COUNT and DUTY Solenoid 7
	DA			H'3FFF'		 ; COUNT and DUTY Solenoid 8
	DA			H'3FFF'		 ; COUNT and DUTY Solenoid 9
	DA			H'3FFF'		 ; COUNT and DUTY Solenoid 10
	DA			H'3FFF'		 ; COUNT and DUTY Solenoid 11
	DA			H'3FFF'		 ; COUNT and DUTY Solenoid 12

	DA			H'3F00'		; natural sounding (randomness) flag initially cleared

; recording region
 	org	H'1020' ; start of recording for flash ending at H19BE (RAM is H2070 to H29AC) All initially erased (H3FFF)


; define reset and interrupt vector start addresses

	org		0  				; start at address 0000h
	goto	MAIN
	org     	4				; interrupt vector 0004h, start interrupt routine here


;***************************************************************************************
INTERRUPT
; Timer1 overflow rate of 10ms (1.333us clock) and 10ms is a count of D7500. So load HFFFF-D7500 or E2B3
	movlb	D'0'
	bcf		PIR1,TMR1IF		; timer 1 overflow flag cleared
	bcf		T1CON,TMR1ON	; timer stopped

	btfss	PLAYING,0			; when set use randomisation. Only during playback
	goto	NON_RNDM
	btfss	NATURAL,0			; when set run randomisation
	goto	NON_RNDM

; modify values randomly
; check random change rate (bits 4,5,6 in NATURAL)
	swapf	NATURAL,w		; place bits 4,5,6,7 into 0,1,2,3
	andlw	B'00000111'
	movwf	WORKING
	movf	WORKING,w
	btfsc	STATUS,Z		; if zero 
	goto	FIVET			; 1 to 5 times variation
	decfsz	WORKING,f		; if was 1
	goto	CKTWO
	goto	THREET			; 1-3 times vary
	
CKTWO
	decfsz	WORKING,f		; if was 2
	goto	CKONEFIVE
	goto	TWOT			; 1-2 times vary
CKONEFIVE
	bcf		AARGB3,4		; 1.5x
TWOT
	bcf		AARGB3,5		; 2x
THREET
	bcf		AARGB3,6		; 3x
FIVET
	bcf		AARGB3,7		; 5 times variation

	movf	AARGB3,w
	sublw	H'E2'			; varies counter rate
	goto	RNDM_ON

NON_RNDM
	movlw	H'E2'
RNDM_ON			; 
	movwf	TMR1H			; ms byte
	movlw	H'B4'			; compensated value for timer off
	movwf	TMR1L			; ls byte
	bsf		T1CON,TMR1ON	; timer started
	bsf		TEN_MS,0		; set for 10ms indicator flag

	btfss	PLAYING,0		; if clear, is in record or calibration mode so bypass LDR check
	goto	RUN_RND		; bypass in record mode

; Check RA1 for a low either through LDR in light or the shunt installed (JP2)
	btfss	PORTA,1 		; LDR light level
	goto	RUN_RND

; flash LED on and off
	movlb	D'2'				; latch, bank 2
	bsf		LATC,1			; LED on
	movlb	D'0'
	movlw	D'255'
	call		DELX
	movlb	D'2'				; latch, bank 2
	bcf		LATC,1			; LED off
	movlb	D'0'
	movlw	D'255'
	call		DELX
	goto	INTERRUPT		; wait for LDR

RUN_RND
; run random generator at random rate between every 10s and at most every 10m (ie 1-127 variation) maximum value. 
; every X interrupts run random generator. X is between 1000 (for 10s) and 640s
; reduce D1000 (H3E8) by AARGB1 times. When zero, re-run random generator  

	decfsz	RNDM2,f
	retfie
	decfsz	RNDM1,f
	retfie

	movlw	H'3'
	movwf	RNDM1			; D1000'
	movlw	H'EB'
	movwf	RNDM2
; reduce AARGB1
RED1
	decfsz	AARGB1,f
	retfie
T1	call		RAND32			; random generator
	movf	AARGB1,w
	movlw	D'1'
	btfsc	STATUS,Z		; if zero set at 1
	movwf	AARGB1		; minimum of 1
	
; check random change rate (bits 1,2,3 in NATURAL)
	movf	NATURAL,w
	andlw	B'00001110'
	movwf	WORKING
	lsrf		WORKING,f		; shift right so is a count from 0-7
	movf	WORKING,w
	btfsc	STATUS,Z		; if zero 
	goto	SF				; 10-1280s
	decfsz	WORKING,f		; if was 1
	goto	SKIP1
	goto	TT				; 10-640s
SKIP1
	decfsz	WORKING,f		; if was 2
	goto	SKIP2
	goto	ST				; 10-320s
SKIP2
	decfsz	WORKING,f		; if was 3
	goto	SKIP3
	goto	ET				; 10-160s
SKIP3
	decfsz	WORKING,f		; if was 4
	goto	TO
	goto	FR				; 10-80s

; working value was 5. for 10-40s

TO	bcf		AARGB1,2		; 4 max 40s
FR	bcf		AARGB1,3		; 8 max 80s
ET	bcf		AARGB1,4		; 16 max 160s
ST	bcf		AARGB1,5		; 32 max 320s
TT	bcf		AARGB1,6		; 64 max (640seconds)
SF	bcf		AARGB1,7		; 128 (1280s)
	movf	AARGB1,w		; make sure it is not zero
	movlw	D'1'
	btfsc	STATUS,Z		; if zero set at 1
	movwf	AARGB1		; minimum of 1

	retfie
		
;..................................................................................................................................

MAIN
; initial values
; set inputs/outputs
	movlb	D'2'			; latch, bank 2
; comparators off
	bsf		CM2CON0,7
	bsf		CM1CON0,7
	
; weak pullups off
	movlb	D'4'			; bank4	WPUA/B
	clrf		WPUA
	clrf		WPUB

; set I/O
	movlb	D'1'			; bank1	TRISA/B/C
	movlw   B'00111011'	; I/O 
	movwf   TRISA		; port A data direction register
	movlw	B'11110000'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlw	B'11111101'
	movwf	TRISC		; port C data direction 
; options
	movlw	B'00000000'	; weak pullups set via WPUA/B
	movwf	OPTION_REG	; 
; analog inputs
	movlb	D'3'			; bank3	ANSELA/B/C
	movlw	B'00000000'	; 
	movwf	ANSELA
	movlw	B'00000000' ; 
	movwf	ANSELB
	movlw	B'00000001'	; 
	movwf	ANSELC
; A/D select
	movlb	D'1'			; bank1	ADCON0/1/2
	movlw	B'00010000'	; channel 4
	movwf	ADCON0
	movlw	B'01100000'	; left justified A/D result, fosc /64, Vdd to Vss A/D
	movwf	ADCON1
	bsf		ADCON0,ADON	; A/D on

; oscillator	
	movlw	B'11111000'	; for 24MHz; 8MHz x 3PLL
	movwf	OSCCON	; osc

	movlb	D'0'			; bank0	
; timer1 set
	movlw	B'00110000'		; 24MHz/4/8;  period 1.333us per count or 87.38131ms for 16-bit count available from flash memory (two words) 
	movwf	T1CON
	bsf		T1CON,TMR1ON	; timer started

; ensure the USB is off as it affects the RA0 and RA1 inputs 
	movlb	D'29'
	bcf		UCON,3			; usb off
	bcf		UCON,1

	movlb	D'2'			; latch, bank 2
; set A , B and C port solenoids output as a high when an output 
	movlw	B'00110000'	; A
	movwf	LATA
	movlw	B'11110000'	; B
	movwf	LATB
	movlw	B'11111100'	; C
	movwf	LATC
	movlb	D'0'				; bank 0

;..............................................................................................................
; For individual solenoid calibration values, COUNT and DUTY are stored in a single word. H1000 for Solenoid 1, H1001 for solenoid 2
; Flash memory has stored recording from H'1020' to H'14BE' (RAM is H2070 to H29AC) 
; Recording format is 14 bits, bit 13 is clear for the drive data and set for the pause between solenoid drives. 

; Flash
; Drive data is in  two words.
; word 1 is the drive data bit 13 is clear, solenoid number (1-12), in ls bits
; word 2 is the Count value in ms 7-bits,  Duty value in ls 7 bits

; Pause data is using two words and 10ms per count ( uses 16-bits count for up to 10.9minutes).   
; word 1 is ms count (8-bits)
; word 2 is ls count (8-bits)

; RAM uses two bytes per flash word
; drive data
; ram 1; bits are clear (8-bits)
; ram 2; solenoid number (1-12) 
; ram 3; count (7-bits)
; ram 4; duty (7-bits)

; Pause data
; ram 1; bits are set (8-bits)
; ram 2; ms PAUSE (8-bits)
; ram 3; bits are set (8-bits)
; ram 4; ls PAUSE (8-bits)

; initial
	clrf		PLAYING		; play flag
	clrf		STARTED_REC	; started to record flag

	call		DELAY
	call		DELAY

; wait for stable oscillator
	movlb	D'01'				; bank 1
HF1
	btfss	OSCSTAT,HFIOFS	; HF internal oscillator is stable
	goto	HF1					; not stable yet
	btfss	OSCSTAT,HFIOFR	; HF internal oscillator ready
	goto	HF1					; not ready yet
	btfss	OSCSTAT,PLLRDY	; phase lock loop ready check
	goto	HF1
	movlb	D'0'					; bank 0

; set initial random generator seed value
	movlw	0x27			; random seed value
	movwf	RANDB0
	movlw	0xF1
	movwf	RANDB1
	movlw	0x20			; random seed value
	movwf	RANDB2
	movlw	0x49
	movwf	RANDB3
	call		RAND32			; random generator
	movlw	D'1'
	movwf	AARGB1		; run random after first run in interrupt

; initialise interrupt counters
	movlw	H'0'
	movwf	RNDM1			; around 1s initially 
	movlw	H'64'
	movwf	RNDM2
	
; interrupts enable
	movlb	D'1'
	bsf		PIE1,TMR1IE		; enable timer1 interrupt
	movlb	D'0'
	bcf		PIR1,TMR1IF		; flag clear
	bsf		INTCON,PEIE	; enable peripheral interrupts
	bsf		INTCON,GIE

; check if the record switch is pressed (denoting pressed at power up)
	btfss	PORTA,3
	goto	CALIBRATE

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
; playback startup delay

	movlw	D'10'
	movwf	TEST_COUNT
EXTRA_DEL
	call		DELAY
	decfsz	TEST_COUNT,f
	goto	EXTRA_DEL

; initially read calibration values and Natural flags
; read flash and place in RAM
; initial RAM location (H2050)
	movlw	H'20'
	movwf	FSR0H			; ms byte
	movlw	H'50'	
	movwf	FSR0L
; Read stored value for COUNT and DUTY for solenoid 
	movlw	H'10'			; ms address byte 
	movwf	READADDMS 
	movlw	H'00'
	movwf	READADDLS	; ls address byte
LOOP_RDA
	call		READ			; ms byte in TEMP_VAL
	movwf	TEMP			; ls byte
	movwi	FSR0++			; ls byte to RAM
	movf	TEMP_VAL,w	; ms byte
	movwi	FSR0++			; ms byte to RAM

; next flash memory read

	incf		READADDLS,f
	movf	READADDLS,w
	xorlw	D'14'			; stop at 13 (only 13 addresses used)	
	btfss	STATUS,Z
	goto	LOOP_RDA

; set address to NATURAL flag location  
	movlw	H'20'
	movwf	FSR0H			; ms byte
	movlw	H'68'	
	movwf	FSR0L

; Read stored value for NATURAL
	movlw	H'10'			; ms address byte 
	movwf	READADDMS 
	movlw	H'0C'
	movwf	READADDLS	; ls address byte
	call		READ			; ls byte in w
	movwf	NATURAL
	movwi	FSR0++			; ls byte to RAM at H2068
; return to NATURAL flag location
	movlw	H'68'	
	movwf	FSR0L

; check start with pressed switch
ONE
	btfsc	PORTA,5		; S1
; clear  NATURAL bit 0 (at H2068')
	bcf		NATURAL,0		; random flag cleared
TWO
; check start with pressed switch
	btfsc	PORTA,4		; S2
; set bit 0 in NATURAL 
	bsf		NATURAL,0		; random on

; randomise change rate bits 1,2,3. Switches 3-8
THREE
	btfss	PORTC,5		; S3, xxx000x
	goto	FOUR
	bcf		NATURAL,1
	bcf		NATURAL,2
	bcf		NATURAL,3
FOUR
	btfss	PORTC,4		; S4, xxx001x
	goto	FIVE
	bsf		NATURAL,1
	bcf		NATURAL,2
	bcf		NATURAL,3
FIVE
	btfss	PORTC,3		; S5, xxx010x
	goto	SIX
	bcf		NATURAL,1
	bsf		NATURAL,2
	bcf		NATURAL,3
SIX
	btfss	PORTC,6		; S6, xxx011x
	goto	SEVEN
	bsf		NATURAL,1
	bsf		NATURAL,2
	bcf		NATURAL,3
SEVEN	
	btfss	PORTC,7		; S7, xxx100x
	goto	EIGHT
	bcf		NATURAL,1
	bcf		NATURAL,2
	bsf		NATURAL,3
EIGHT
	btfss	PORTB,7		; S8, xxx101x
	goto	NINE
	bsf		NATURAL,1
	bcf		NATURAL,2
	bsf		NATURAL,3

; randomise range rate bits 4,5,6. Switches 9-12

NINE
	btfss	PORTC,2		; S9, 000xxxx
	goto	TEN
	bcf		NATURAL,4
	bcf		NATURAL,5
	bcf		NATURAL,6
TEN
	btfss	PORTB,6		; S10, 001xxxx
	goto	ELEVEN
	bsf		NATURAL,4
	bcf		NATURAL,5
	bcf		NATURAL,6
ELEVEN 
	btfss	PORTB,5		; S11, 010xxxx
	goto	TWELVE
	bcf		NATURAL,4
	bsf		NATURAL,5
	bcf		NATURAL,6
TWELVE; clears range	
	btfss	PORTB,4		; S12, 011xxxx
	goto	VALUES
	bsf		NATURAL,4
	bsf		NATURAL,5
	bcf		NATURAL,6

VALUES; store random values
; acknowledge
	movlb	D'2'				; latch, bank 2
	bsf		LATC,1			; LED on
	movlb	D'0'
	call		DELAY
	call		DELAY
	call		DELAY
	movlb	D'2'				; latch, bank 2
	bcf		LATC,1			; LED off
	movlb	D'0'

	movf	NATURAL,w
	movwi	FSR0++			; load value to ram	
	call		FLASH_SET		; store in Flash

PLAY
RESTART
	bsf		PLAYING,0		; set play flag so randomisation runs in play only

; Read Natural flags
	movlw	H'10'			; ms address byte 
	movwf	READADDMS 
	movlw	H'0C'
	movwf	READADDLS	; ls address byte
	call		READ			; TEMP_VAL has ms bits, w has ls byte
	movwf	NATURAL		; flags available to read

; Read stored values starting at beginning of recording
	movlw	H'10'			; ms address byte 
	movwf	READADDMS 
	movlw	H'20'
	movwf	READADDLS	; ls address byte

CYCLE_THROUGH

; check if the record switch is pressed (denoting change to record)
	btfss	PORTA,3
	goto	RECORDS

	call		READ			;  TEMP_VAL has ms bits, w has ls byte
	movwf	TEMP			; ls byte

	btfsc	TEMP_VAL,5	; check bit 5. If set then data is the pause period
	goto	PAUSE_PERIOD

; drive
	movwf	SOLENOID		; ls byte of read has solenoid number 1 to 12

; get COUNT and DUTY from next memory location
	incfsz	READADDLS,f
	goto	BY_MS_INC
	incf		READADDMS,f 
BY_MS_INC
	call		READ			; Temp_val has ms 6-bits 'w' has ls 8-bits
	movwf	TEMP			; temporary working

	call		EXTRACT		; get DUTY and COUNT
	movf	COUNT,w
	movwf	COUNT_VAL		; store value
PWM_ON
	call		DRV_S			; drive solenoid
	movlb	D'2'				; latch, bank 2
	bsf		LATC,1			; LED on
	movlb	D'0'
; on for 256-DUTY
	movf	DUTY,w
	call		DELY			;  on for DUTY
	call		DRV_OFF
	movlb	D'2'				; latch, bank 2
	bcf		LATC,1			; LED off
	movlb	D'0'

PWM_OFF
	movf	DUTY,w
	sublw	D'129'
	call		DELY		; 

	decfsz	COUNT,f		; drive duration
	goto	PWM_ON

CHECK_END
; increase memory address and check for end at H14BE
	incfsz	READADDLS,f
	goto	BY_MS_INC1
	incf		READADDMS,f
BY_MS_INC1
; compare with H14BE
	movf	READADDMS,w
	xorlw	H'14'
	btfss	STATUS,Z	
 	goto	CYCLE_THROUGH
	movf	READADDLS,w
	sublw	H'BE'
	btfsc	STATUS,Z
	goto	RESTART
	btfss	STATUS,C
	goto	RESTART
	goto	CYCLE_THROUGH

PAUSE_PERIOD

; check if TEMP_VAL is H'33'. This is the end of recording flag. If so then return to start.
	movf	TEMP_VAL,w
	xorlw	H'33'
	btfsc	STATUS,Z
	goto	RESTART
; period between solenoid drives
	
	incfsz	READADDLS,f
	goto	BY_MS_INC2
	incf		READADDMS,f 
BY_MS_INC2
	call		READ			; 
	movwf	TEMP_VAL

; subtract 100ms from pause value in TEMP_VAL and TEMP to compensate for 100ms switch debounce (in 10ms steps so subtract value is 10)
; subtract solenoid on period. This is in COUNT_VAL
; conversion to 10ms steps is ideally a divide by 5. Approximation is first divide by 4 for value1, then divide by 16 for value2. Subtract value2 from value1  for close enough to divide by 5. 
; Approximation can be up to 20ms out compared to a true /5 but not noticeable.
 
	lsrf		COUNT_VAL,f	; /2
	lsrf		COUNT_VAL,f	; /4

	movf	COUNT_VAL,w
	movwf	COUNT_STO	; store
	lsrf		COUNT_VAL,f	; /8
	lsrf		COUNT_VAL,w	; /16
	subwf	COUNT_STO,w	; subtract /16 from /4

; Take away from pause value
	subwf	TEMP,f
	movlw	D'1'
	btfss	STATUS,C
	subwf	TEMP_VAL,f
	btfss	STATUS,C
	clrf		TEMP_VAL		; set at 0 if carry is clear
; add value to compensate for switch debounce
	movlw	D'8';10'
	addwf	TEMP,f
	btfsc	STATUS,C		; if carry increase ms byte
	incfsz	TEMP_VAL,f
	goto	TENMILLIS
	movlw	H'FF'
	movwf	TEMP_VAL		; if over keep at FF
	movwf	TEMP 
	
; wait for 10ms
TENMILLIS
	clrf		TEN_MS		; cleared, bit 0 set after 10ms
WAIT_10

; check if the record switch is pressed (denoting change to record)
	btfss	PORTA,3
	goto	RECORDS

	btfss	TEN_MS,0		; when set 10ms period 
	goto	WAIT_10

; decrease TEMP_VAL and TEMP
; check if zero
	movf	TEMP_VAL,w	; ms byte of counter	
	btfss	STATUS,Z		; if zero check ls byte
	goto	DEC_TMP
	movf	TEMP,w
	btfsc	STATUS,Z
	goto	CHECK_END	; time period end

DEC_TMP

	movf	TEMP,f			; ls byte
	btfss	STATUS,Z
	goto	REDLS			; if not zero then bypass ms byte
; if ls byte is zero, check if ms byte is zero
	movf	TEMP_VAL,w
	btfsc	STATUS,Z		; if zero then end
	goto	CHECK_END
	decf	TEMP_VAL,f		; decrease counter ms byte
REDLS; reduce ls byte
	decf	TEMP,f
	goto	TENMILLIS
	
;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
RECORDS
	clrf		PLAYING		; not play mode
	clrf		STARTED_REC	; record started flag, cleared
	movlb	D'2'				; latch, bank 2
	bsf		LATC,1			; LED on
	movlb	D'0'

; wait for switch off
OFFS
	call		DELAY			; 100ms
	btfss	PORTA,3
	goto	OFFS

; clear all RAM
; initial RAM location (H2070)
	movlw	H'20'
	movwf	FSR0H			; ms byte
	movlw	H'70'	
	movwf	FSR0L
 ; (set at FF)
LOOP_ERASE
	movlw	H'FF'
	movwi	FSR0++

; compare with H29AC (using value divisible by 4)	
 	movf		FSR0H,w
	sublw		H'29'
	movwf		TEMP
	movf		FSR0L,w
	sublw		H'AC'
	btfss		STATUS,C
	decf		TEMP,f
	btfss		TEMP,7
	goto		LOOP_ERASE

; initial RAM location (H2070)
	movlw	H'20'
	movwf	FSR0H			; ms byte
	movlw	H'70'	
	movwf	FSR0L

;  clear timer counters
	clrf		TIMER_MS		; timer for pause between solenoids ms byte
	clrf		TIMER_LS		; timer ls byte

RECORD1
CONT_RECORD
; check record switch
	btfsc	PORTA,3	
	goto	CONT

; short press <1s then end recording
; start counting in 10ms increments
	clrf		VALUE_1		; counter

LP10
	clrf		TEN_MS		; 10ms flag
STAYX
	btfss	TEN_MS,0		; when set then 10ms is up
	goto	STAYX
	incf		VALUE_1,f		; timer increased each 10ms
; check after 100ms if switch is open
	movf	VALUE_1,w
	sublw	D'9'	
	btfsc	STATUS,C
	goto	LP10
	btfss	PORTA,3		; if set then switch is open
	goto	SW_CL	
; switch open	 
; if switch opens when VALUE_1 is 100 or less, then store and play
	movf	VALUE_1,w
	sublw	D'100'	
	btfsc	STATUS,C
	goto	STORE_PLAY	; store in flash and revert to play
	goto	LONGER

SW_CL
; if switch closed when VALUE_1 is 100 then goto LONGER
	movf	VALUE_1,w
	sublw	D'100'	
	btfsc	STATUS,C
	goto	LP10
;..................................................................................................................................
LONGER
	call		STO_TIME		; store time period already expired

; longer press >1s then increase counter at 10s per second  increase by 10 each 10ms

; 100 count. so start rapid count for 10x (1s press for 10s increment)
; initialise LED flashing
	movlw	D'5'
	movwf	FLASH_LED	
	movlb	D'2'				; latch, bank 2
	bcf		LATC,1			; LED off
	movlb	D'0'	
LONG
	decfsz	FLASH_LED,f	; when zero reload value and light or switch off LED
	goto	CHK_REC
	movlw	D'50'
	movwf	FLASH_LED	
; check if LED on or off
	movlb	D'2'				; latch, bank 2
	btfss	LATC,1			; LED state
	goto	LED_FLASH
; flash LED at 1s rate
	
	bcf		LATC,1			; LED off
	movlb	D'0'	
	goto	CHK_REC
LED_FLASH ; on
	bsf		LATC,1			; LED on
	movlb	D'0'	

CHK_REC
; check record switch
	btfss	PORTA,3
	goto	COUNT_10S
	movlb	D'2'				; latch, bank 2
	bsf		LATC,1			; LED on (keep LED on)
	movlb	D'0'	
	goto	CONT

COUNT_10S
	clrf		TEN_MS		; 10ms flag
STAY_L
	btfss	TEN_MS,0		; when set then 10ms is up
	goto	STAY_L

	movlw	D'10'
	addwf	TIMER_LS,f		; timer increased each 10ms
	btfss	STATUS,C
	goto	LONG
	incfsz	TIMER_MS,f		; if overrange increase RAM
	goto	LONG

; overrange
	movlw		H'FF'			; TIMER_LS
	movwi		FSR0++	
	movlw		B'11111111'		; for ms byte to set bit 5 of flash for pause flag
	movwi		FSR0++
	movlw		H'FF'			; TIMER_MS
	movwi		FSR0++
	movlw		B'11111111'
	movwi		FSR0++

; compare with H29AC (using value divisible by 4)	
 	movf		FSR0H,w
	sublw		H'29'
	movwf		TEMP
	movf		FSR0L,w
	sublw		H'AC'
	btfss		STATUS,C
	decf		TEMP,f
	btfss		TEMP,7
	goto		LONG

; place end of recording flags H'33'
	goto		PLACE_END

;....................................................................................................................................	

STORE_PLAY
	movlb	D'2'				; latch, bank 2
	bcf		LATC,1			; LED off
	movlb	D'0'

; if started recording then store in FLASH
	btfss	STARTED_REC,0
	goto	OFF1

	call		STO_TIME		; stores time period from last solenoid
; place end of recording flags H'33'
	movf		TIMER_LS,w
	movwi		FSR0++	
	movlw		H'33'
	movwi		FSR0++	
	movf		TIMER_MS,w
	movwi		FSR0++
	movlw		H'33'
	movwi		FSR0++

END_RAM ; from when ram is ended (H29AC) or when end of recording with S13 pressed
	call		STO_FLASH
; wait for switch off
OFF1
	call		DELAY			; 100ms	
	btfss	PORTA,3
	goto	OFF1
	goto	PLAY	

CONT
 ; continue with record
	btfss	STARTED_REC,0
	goto	SWITCH_SOL		; wait for a solenoid switch

; record time period. If overrange go to next RAM locations
; increase time count (if overrange store value in RAM and increase RAM)
; clear TEN_MS flag and wait for flag
; run SWITCH_SOL (SOL for solenoid)
; in switch pressed section store delay period and clear time period count (also cleared at start of record)

	clrf		TEN_MS		; 10ms flag
STAY
	btfss	TEN_MS,0		; when set then 10ms is up
	goto	STAY
	incfsz	TIMER_LS,f		; timer increased each 10ms
	goto	SWITCH_SOL
	incfsz	TIMER_MS,f		; if overrange increase RAM
	goto	SWITCH_SOL

; overrange
	movlw		H'FF'			; TIMER_LS
	movwi		FSR0++	
	movlw		B'11111111'		; for ms byte to set bit 5 of flash for pause flag
	movwi		FSR0++
	movlw		H'FF'			; TIMER_MS
	movwi		FSR0++
	movlw		B'11111111'
	movwi		FSR0++

BY_MS1
; compare with H29AC (using value divisible by 4)	
 	movf		FSR0H,w
	sublw		H'29'
	movwf		TEMP
	movf		FSR0L,w
	sublw		H'AC'
	btfss		STATUS,C
	decf		TEMP,f
	btfss		TEMP,7
	goto		SWITCH_SOL

; place end of recording flags H'33'
PLACE_END
	movlw		H'29'
	movwf		FSR0H			; ms byte
	movlw		H'A9'	
	movwf		FSR0L
	
	movlw		H'FF'			; TIMER_LS
	movwi		FSR0++	
	movlw		H'33'
	movwi		FSR0++	
	movlw		H'FF'			; TIMER_MS
	movwi		FSR0++
	movlw		H'33'
	movwi		FSR0++
	
	goto		END_RAM
		
; wait for a solenoid switch (S1-S12) to start
SWITCH_SOL
; check PORTA
	movf	PORTA,w
	movwf	PORTA_STO
	andlw	B'00110000'		; get RA4 and RA5
	btfss	STATUS,Z		; if zero then no switch pressed
	goto	A_CHECK		; check which switch in PORTA	
; PortB switches 
	movf	PORTB,w
	movwf	PORTB_STO
	andlw	B'11110000'		; get RB4 to RB7
	btfss	STATUS,Z		; if zero then no switch pressed
	goto	B_CHECK		; check which switch in PORTB	
; PortC switches
	movf	PORTC,w
	movwf	PORTC_STO
	andlw	B'11111100'		; get RC2 to RC7
	btfsc	STATUS,Z		; if zero then no switch pressed
	goto	CONT_RECORD 	; 	
C_CHECK	
; which switch in PORTC
	btfsc	PORTC_STO,7
	goto	SOL_7			; solenoid 7
	btfsc	PORTC_STO,6
	goto	SOL_6			; solenoid 6	
	btfsc	PORTC_STO,5
	goto	SOL_3			; solenoid 3
	btfsc	PORTC_STO,4
	goto	SOL_4			; solenoid 4	
	btfsc	PORTC_STO,3
	goto	SOL_5			; solenoid 5
	btfsc	PORTC_STO,2
	goto	SOL_9			; solenoid 9
	goto	CONT_RECORD 	; false switch detection 
B_CHECK
	btfsc	PORTB_STO,7
	goto	SOL_8			; solenoid 8
	btfsc	PORTB_STO,6
	goto	SOL_10			; solenoid 10
	btfsc	PORTB_STO,5
	goto	SOL_11			; solenoid 11
	btfsc	PORTB_STO,4
	goto	SOL_12			;  solenoid 12
	goto	CONT_RECORD 	; false switch detection 
A_CHECK
	btfsc	PORTA_STO,5
	goto	SOL_1			; solenoid 1
	btfsc	PORTA_STO,4
	goto	SOL_2			; solenoid 2
	goto	CONT_RECORD 	; false switch detection 

;...............................................................................................................................................
; Subroutines

EXTRACT
; extract COUNT and DUTY
	movf	TEMP,w			; ls byte (bits 0-6) are the DUTY value
	movwf	DUTY
	bcf		DUTY,7			; bit 7 has ls bit of COUNT, so clear bit
; make sure it is not 0
	movf	DUTY,w
	movlw	D'1'
	btfsc	STATUS,Z
	movwf	DUTY

; get COUNT value
	rlf		TEMP,w			; get ms bit to the carry
	rlf		TEMP_VAL,w	; carry shifted in
	movwf	COUNT
	bcf		COUNT,7		; ms bit cleared so 7-bit data
; make sure it is not 0
	movf	COUNT,w
	movlw	D'1'
	btfsc	STATUS,Z
	movwf	COUNT
	return
;.......................................................................................................................................
STO_FLASH ; store recording into flash memory
; Store in program memory 
	movlw	H'10'
	movwf	ADDRH			; upper byte flash address
	movlw	H'20'	;
	movwf	ADDRL			; lower byte 
; start of RAM
	movlw	H'20'
	movwf	FSR0H
	movlw 	H'70'
	movwf	FSR0L

; end when ADDRH,L is 14BE
LOOPWR
	call		WRITE

; add H20 to ADDRL,H ; check value ADDRH,L (programmed in 32byte steps)
	movlw	H'20'
	addwf	ADDRL,f
	btfsc	STATUS,C	; if carry, add 1 to ADDRH
	incf		ADDRH,f

	movf	ADDRH,w
	xorlw	H'14'		; 
	btfss	STATUS,Z; 
	goto	LOOPWR
; check fo if greater than BE
	movf	ADDRL,w
	sublw	H'BE'
	btfsc	STATUS,C
	goto	LOOPWR
; end of store in program memory
	return

;........................................................................................................................................
CALIBRATE
	clrf		PLAYING		; not play mode
	clrf		SOLENOID		; set at zero so none is initially selected
; acknowledge
	movlb	D'2'				; latch, bank 2
	bsf		LATC,1			; LED on
	movlb	D'0'
	call		DELAY			; 100ms
	call		DELAY
	movlb	D'2'				; latch, bank 2
	bcf		LATC,1			; LED off
	movlb	D'0'
	call		DELAY			; 100ms
	call		DELAY
	movlb	D'2'				; latch, bank 2
	bsf		LATC,1			; LED on
	movlb	D'0'
	call		DELAY			; 100ms

; read COUNT and DUTY values for solenoids 1-12 and place in the RAM
; from flash at H1000 to 101F to RAM at H2050 to 206F (only first 12 flash memories are used for used data). remaining just add up to 32 for separate programming easiness. 
; so solenoid 1 uses RAM H2050 and Flash H1000 for COUNT (on period). RAM H2051 and flash H1000 for DUTY (PWM) with flash holding the two 7-bit values 
; Natural (random flag) is 13th flash location

; read flash and place in RAM
; initial RAM location (H2050)
	movlw	H'20'
	movwf	FSR0H			; ms byte
	movlw	H'50'	
	movwf	FSR0L
; Read stored value for COUNT and DUTY for solenoid 
	movlw	H'10'			; ms address byte 
	movwf	READADDMS 
	movlw	H'00'
	movwf	READADDLS	; ls address byte
LOOP_RD
	call		READ			; ms byte in TEMP_VAL
	movwf	TEMP			; ls byte
	movwi	FSR0++			; ls byte to RAM
	movf	TEMP_VAL,w	; ms byte
	movwi	FSR0++			; ms byte to RAM
; next flash memory read

	incf		READADDLS,f
	movf	READADDLS,w
	xorlw	D'14'			; stops at 13 (only 13 addresses used)	
	btfss	STATUS,Z
	goto	LOOP_RD

RDY_CAL
SWITCH_X; switches check closure

; end of calibration (long press of S13) signals a write to Flash for all

; short press of S13 is for driving solenoid selected (SOLENOID) has solenoid number (1-12)

	clrf		SWITCH			; S13 flag

	btfsc	PORTA,3		; switch pressed when low
	goto	OTHER_SW

; if SOLENOID is zero then bypass
	movf	SOLENOID,w
	btfsc	STATUS,Z
	goto	OTHER_SW		; solenoid switches

; set switch 13 flag
	bsf		SWITCH,0

	movf	SOLENOID,w
	movwf	TEMP
H1	decfsz	TEMP,f
	goto	H2
	goto	SOL_1_CAL
H2	decfsz	TEMP,f
	goto	H3
	goto	SOL_2_CAL
H3 	decfsz	TEMP,f
	goto	H4
	goto	SOL_3_CAL
H4	decfsz	TEMP,f
	goto	H5
	goto	SOL_4_CAL
H5	decfsz	TEMP,f
	goto	H6
	goto	SOL_5_CAL
H6	decfsz	TEMP,f
	goto	H7
	goto	SOL_6_CAL
H7 	decfsz	TEMP,f
	goto	H8
	goto	SOL_7_CAL
H8	decfsz	TEMP,f
	goto	H9
	goto	SOL_8_CAL
H9	decfsz	TEMP,f
	goto	H10
	goto	SOL_9_CAL
H10	decfsz	TEMP,f
	goto	H11
	goto	SOL_10_CAL
H11	decfsz	TEMP,f
	goto	SOL_12_CAL
	goto	SOL_11_CAL

; switch press
PRESS	
; extract COUNT and DUTY values for drive
	movf	TEMP_RAMMS,w	; ms byte
	movwf	TEMP_VAL		; ms byte to temp;
 
	movf	TEMP_RAMLS,w	; ls byte
	movwf	TEMP
	call		EXTRACT		; get COUNT and DUTY values
; drive solenoid based on SOLENOID value
; if SOLENOID is zero then bypass
	movf	SOLENOID,w
	btfsc	STATUS,Z
	goto	OTHER_SW		; solenoid switches

PWM_ON_C
	movlb	D'2'				; latch, bank 2
	bsf		LATC,1			; LED on
	movlb	D'0'
	call		DRV_S			; drive solenoid

; on for DUTY
	movf	DUTY,w
	call		DELY			; on for DUTY
	movlb	D'2'				; latch, bank 2
	bcf		LATC,1			; LED off
	movlb	D'0'
	call		DRV_OFF

PWM_OFF_C
; off for 129-DUTY
	movf	DUTY,w
	sublw	D'129'
	call		DELY		; 

	decfsz	COUNT,f		; drive duration
	goto	PWM_ON_C

; if switch still pressed after delay then write to flash
	movlw	D'10'
	movwf	SW_HOLD
SW_LOOP
	call		DELAY			; 100ms
	btfsc	PORTA,3		; switch closed?
	goto	SWITCH_X
	decfsz	SW_HOLD,f
	goto	SW_LOOP

	movlb	D'2'				; latch, bank 2
	bsf		LATC,1			; LED on
	movlb	D'0'

OP	call		DELAY			; 100ms
; wait for switch open
	btfss	PORTA,3
	goto	OP	
	movlb	D'2'				; latch, bank 2
	bcf		LATC,1			; LED off
	movlb	D'0'
	call		FLASH_SET		; write the calibration values to flash 
	goto	SWITCH_X

; wait for solenoid switch
OTHER_SW
; check PORTA
	movf	PORTA,w
	movwf	PORTA_STO
	andlw	B'00110000'		; get RA4 and RA5
	btfss	STATUS,Z		; if zero then no switch pressed
	goto	A_CHECK_CAL	; check which switch in PORTA	
; PortB switches 
	movf	PORTB,w
	movwf	PORTB_STO
	andlw	B'11110000'		; get RB4 to RB7
	btfss	STATUS,Z		; if zero then no switch pressed
	goto	B_CHECK_CAL	; check which switch in PORTB	
; PortC switches
	movf	PORTC,w
	movwf	PORTC_STO
	andlw	B'11111100'		; get RC2 to RC7
	btfsc	STATUS,Z		; if zero then no switch pressed
	goto	RDY_CAL		; 	
C_CHECK_CAL	
; which switch in PORTC
	btfsc	PORTC_STO,7
	goto	SOL_7_CAL		; solenoid 7
	btfsc	PORTC_STO,6
	goto	SOL_6_CAL		; solenoid 6	
	btfsc	PORTC_STO,5
	goto	SOL_3_CAL		; solenoid 3
	btfsc	PORTC_STO,4
	goto	SOL_4_CAL		; solenoid 4	
	btfsc	PORTC_STO,3
	goto	SOL_5_CAL		; solenoid 5
	btfsc	PORTC_STO,2
	goto	SOL_9_CAL		; solenoid 9
	goto	RDY_CAL		; false switch detection 
B_CHECK_CAL
	btfsc	PORTB_STO,7
	goto	SOL_8_CAL		; solenoid 8
	btfsc	PORTB_STO,6
	goto	SOL_10_CAL	; solenoid 10
	btfsc	PORTB_STO,5
	goto	SOL_11_CAL	; solenoid 11
	btfsc	PORTB_STO,4
	goto	SOL_12_CAL	;  solenoid 12
	goto	RDY_CAL		; false switch detection 
A_CHECK_CAL
	btfsc	PORTA_STO,5
	goto	SOL_1_CAL		; solenoid 1
	btfsc	PORTA_STO,4
	goto	SOL_2_CAL		; solenoid 2
	goto	RDY_CAL		; false switch detection 

SOL_1_CAL
; initial RAM location (H2050,2051) for Solenoid 1 (6-ls bits in 2050 are the count value, 2051 is 8 bits comprising ls bit of COUNT in bit 7, remaining is DUTY value)

; set solenoid number
	movlw	D'1'
	movwf	SOLENOID		; set solenoid number	
	movlw	H'50'	
	goto	ADJUST			; adjust COUNT or DUTY

SOL_2_CAL
; set solenoid number
	movlw	D'2'
	movwf	SOLENOID		; set solenoid number
	movlw	H'52'	
	goto	ADJUST			; adjust COUNT or DUTY

SOL_3_CAL
; set solenoid number
	movlw	D'3'
	movwf	SOLENOID		; set solenoid number
	movlw	H'54'	
	goto	ADJUST			; adjust COUNT or DUTY

SOL_4_CAL
; set solenoid number
	movlw	D'4'
	movwf	SOLENOID		; set solenoid number	
	movlw	H'56'	
	goto	ADJUST			; adjust COUNT or DUTY

SOL_5_CAL
; set solenoid number
	movlw	D'5'
	movwf	SOLENOID		; set solenoid number	
	movlw	H'58'	
	goto	ADJUST			; adjust COUNT or DUTY

SOL_6_CAL
; set solenoid number
	movlw	D'6'
	movwf	SOLENOID		; set solenoid number
	movlw	H'5A'	
	goto	ADJUST			; adjust COUNT or DUTY

SOL_7_CAL
; set solenoid number
	movlw	D'7'
	movwf	SOLENOID		; set solenoid number	
	movlw	H'5C'	
	goto	ADJUST			; adjust COUNT or DUTY

SOL_8_CAL
; set solenoid number
	movlw	D'8'
	movwf	SOLENOID		; set solenoid number	
	movlw	H'5E'	
	goto	ADJUST			; adjust COUNT or DUTY

SOL_9_CAL
; set solenoid number
	movlw	D'9'
	movwf	SOLENOID		; set solenoid number	
	movlw	H'60'	
	goto	ADJUST			; adjust COUNT or DUTY

SOL_10_CAL
; set solenoid number
	movlw	D'10'
	movwf	SOLENOID		; set solenoid number
	movlw	H'62'	
	goto	ADJUST			; adjust COUNT or DUTY

SOL_11_CAL
; set solenoid number
	movlw	D'11'
	movwf	SOLENOID		; set solenoid number
	movlw	H'64'	
	goto	ADJUST			; adjust COUNT or DUTY

SOL_12_CAL
; set solenoid number
	movlw	D'12'
	movwf	SOLENOID		; set solenoid number
	movlw	H'66'	

ADJUST	 ; adjust COUNT or DUTY
	movwf	FSR0L
	movwf	FSR0L_TMP
; set common address
	movlw	H'20'
	movwf	FSR0H			; ms byte

; first get values for packed 2 x 7bit values (COUNT and DUTY)
	moviw	FSR0++			; ls byte to RAM
	movwf	TEMP_RAMLS	; ls byte
	moviw	FSR0++			; ms byte to RAM
	movwf	TEMP_RAMMS	; ms byte

;  read A/D and place COUNT or DUTY, then into solenoid RAM for any pressed switch for a solenoid
; read VR1 for rate
	call		ACQUIRE_AD	; w has digital value
; result in w
	movwf	TEMP
	lsrf		TEMP,w			; 0-127
; check JP1 if in or out. RA0	
	btfss	PORTA,0
	goto	DUTY_STO2

	movwf	COUNT			; make sure it is not zero for solenoid on period	
;check if zero
	movf	COUNT,w
	movlw	D'1'
	btfsc	STATUS,Z
	movwf	COUNT			; make sure it is not zero	

; process for packed 2 x 7-bit data
; clear bit 7 of LS RAM
	bcf		TEMP_RAMLS,7	; ls byte has bit 7 as the ls bit for COUNT
; shift COUNT right
	lsrf		COUNT,w
	movwf	TEMP_RAMMS	; to ms byte
; if carry was set in the shift, then set the ls byte bit 7
	btfsc	STATUS,C
	bsf		TEMP_RAMLS,7
	goto	STO_FSR
	
DUTY_STO2
	movwf	DUTY			;  PWM period
;check if zero
	movf	DUTY,w
	movlw	D'1'
	btfsc	STATUS,Z
	movwf	DUTY			; make sure it is not zero
; compress into a single 14-bit word
; shift TEMP_RAMMS left to place full COUNT value into this memory
	lslf		TEMP_RAMMS,f
; if bit 7 is set for  TEMP_RAMLS, set bit zero in TEMP_RAMMS
	btfsc	TEMP_RAMLS,7			
	bsf		TEMP_RAMMS,0
; place DUTY to TEMP_RAMLS
	movf	DUTY,w
	movwf	TEMP_RAMLS
; shift TEMP_RAMMS right again, set bit 7 in TEMP_RAMLS if carry is set in the shift
	lsrf		TEMP_RAMMS,f
	btfsc	STATUS,C
	bsf		TEMP_RAMLS,7	
STO_FSR
; place values in the correct RAM
	movf	FSR0L_TMP,w	; stored FSR0L
	movwf	FSR0L			; FSR value

	movf	TEMP_RAMLS,w	; ls byte
	movwi	FSR0++			; ls byte to RAM
	movf	TEMP_RAMMS,w	; ms byte
	movwi	FSR0++			; ms byte to RAM

; if from S13 
	btfss	SWITCH,0
	goto	ACK
	clrf		SWITCH
	call		DELAY			; 100ms
	movlb	D'2'				; latch, bank 2
	bcf		LATC,1			; LED off
	movlb	D'0'
	goto	PRESS

ACK
; acknowledge
	movlb	D'2'				; latch, bank 2
	bsf		LATC,1			; LED on
	movlb	D'0'
	call		DELAY			; 100ms
	call		DELAY			; 100ms
	call		DELAY			; 100ms
	movlb	D'2'				; latch, bank 2
	bcf		LATC,1			; LED off
	movlb	D'0'
	goto	SWITCH_X
	
;........................................................................................................

DRV_S
; drive solenoid
	movf	SOLENOID,w
	movwf	TEMP
	btfsc	STATUS,Z	; if zero then no solenoid
	return
; set output associated with solenoid
	decfsz	TEMP,f
	goto	TWO_ON	; 
ONE_ON
; set PORTA,5 as an output Solenoid 1
	movlb	D'1'			; bank1	TRISA/B/C
	movlw   B'00011011'	; I/O 
	movwf   TRISA		; port A data direction register
	movlb	D'0'
	return
TWO_ON
	decfsz	TEMP,f
	goto	THREE_ON
; set PORTA,4 as an output Solenoid 2
	movlb	D'1'			; bank1	TRISA/B/C
	movlw   B'00101011'	; I/O 
	movwf   TRISA		; port A data direction register
	movlb	D'0'
	return
THREE_ON
	decfsz	TEMP,f
	goto	FOUR_ON
; set PORTC,5 as an output	; solenoid 3
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11011101'
	movwf	TRISC		; port C data direction 
	movlb	D'0'	
	return
FOUR_ON
	decfsz	TEMP,f
	goto	FIVE_ON
; set PORTC,4 as an output	; solenoid 4
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11101101'
	movwf	TRISC		; port C data direction 
	movlb	D'0'	
	return
FIVE_ON
	decfsz	TEMP,f
	goto	SIX_ON
; set PORTC,3 as an output	; solenoid 5
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11110101'
	movwf	TRISC		; port C data direction 
	movlb	D'0'	
	return
SIX_ON
	decfsz	TEMP,f
	goto	SEVEN_ON
; set PORTC,6 as an output	; solenoid 6
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'10111101'
	movwf	TRISC		; port C data direction 
	movlb	D'0'	
	return
SEVEN_ON
	decfsz	TEMP,f
	goto	EIGHT_ON
; set PORTC,7 as an output	; solenoid 7
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'01111101'
	movwf	TRISC		; port C data direction 
	movlb	D'0'	
	return
EIGHT_ON
	decfsz	TEMP,f
	goto	NINE_ON
; set PORTB,7 as an output; solenoid 8
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'01110000'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlb	D'0'
	return
NINE_ON
	decfsz	TEMP,f
	goto	TEN_ON
; set PORTC2 as an output; solenoid 9
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11111001'
	movwf	TRISC		; port C data direction 
	movlb	D'0'	
	return
TEN_ON
	decfsz	TEMP,f
	goto	ELEVEN_ON
; set PORTB,6 as an output; solenoid 10
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'10110000'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlb	D'0'
	return
ELEVEN_ON
	decfsz	TEMP,f
	goto	TWELVE_ON
; set PORTB,5 as an output; solenoid 11
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11010000'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlb	D'0'
	return
TWELVE_ON
; set PORTB,4 as an output; solenoid 12
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11100000'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlb	D'0'
	return

DRV_OFF
; drive solenoid off
	movf	SOLENOID,w
	movwf	TEMP
	btfsc	STATUS,Z	; if zero then no solenoid	
	return
; set output associated with solenoid
	decfsz	TEMP,f
	goto	TWO_OFF
ONE_OFF
; set PORTA,5 as an input Solenoid 1
	movlb	D'1'			; bank1	TRISA/B/C
	movlw   B'00111011'	; I/O 
	movwf   TRISA		; port A data direction register
	movlb	D'0'
	return
TWO_OFF
	decfsz	TEMP,f
	goto	THREE_OFF
; set PORTA,4 as an input Solenoid 2
	movlb	D'1'			; bank1	TRISA/B/C
	movlw   B'00111011'	; I/O 
	movwf   TRISA		; port A data direction register
	movlb	D'0'
	return
THREE_OFF
	decfsz	TEMP,f
	goto	FOUR_OFF
; set PORTC,5 as an input	; solenoid 3
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11111101'
	movwf	TRISC		; port C data direction 
	movlb	D'0'	
	return
FOUR_OFF
	decfsz	TEMP,f
	goto	FIVE_OFF
; set PORTC,4 as an input	; solenoid 4
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11111101'
	movwf	TRISC		; port C data direction 
	movlb	D'0'	
	return
FIVE_OFF
	decfsz	TEMP,f
	goto	SIX_OFF
; set PORTC,3 as an input	; solenoid 5
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11111101'
	movwf	TRISC		; port C data direction 
	movlb	D'0'	
	return
SIX_OFF
	decfsz	TEMP,f
	goto	SEVEN_OFF
; set PORTC,6 as an intput	; solenoid 6
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11111101'
	movwf	TRISC		; port C data direction 
	movlb	D'0'	
	return
SEVEN_OFF
	decfsz	TEMP,f
	goto	EIGHT_OFF
; set PORTC,7 as an input	; solenoid 7
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11111101'
	movwf	TRISC		; port C data direction 
	movlb	D'0'	
	return
EIGHT_OFF
	decfsz	TEMP,f
	goto	NINE_OFF
; set PORTB,7 as an input; solenoid 8
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11110000'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlb	D'0'
	return
NINE_OFF
	decfsz	TEMP,f
	goto	TEN_OFF
; set PORTC2 as an input; solenoid 9
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11111101'
	movwf	TRISC		; port C data direction 
	movlb	D'0'	
	return
TEN_OFF
	decfsz	TEMP,f
	goto	ELEVEN_OFF
; set PORTB,6 as an input; solenoid 10
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11110000'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlb	D'0'
	return
ELEVEN_OFF
	decfsz	TEMP,f
	goto	TWELVE_OFF
; set PORTB,5 as an input; solenoid 11
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11110000'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlb	D'0'
	return
TWELVE_OFF
; set PORTB,4 as an input; solenoid 12
	movlb	D'1'			; bank1	TRISA/B/C
	movlw	B'11110000'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlb	D'0'
	return

;.....................................................................................................................................

SOL_1
	btfsc	STARTED_REC,0
	call		STO_TIME		; store time delay from previous gap period
	bsf		STARTED_REC,0
; store solenoid number
	movlw	D'1'	
	movwi	FSR0++	
	movlw	B'00000000'		; bit 5 cleared for drive solenoid
	movwi	FSR0++
; wait for switch open
	call		DELAY			; 100ms debounce
S1_O
	movlw	D'1'
	call		DELAY_SET		; 10ms 
	btfsc	PORTA,5
	goto	S1_O
; Read stored value for COUNT and DUTY for solenoid 1
	movlw	H'00'	; ms byte address, STOX does read
	goto	STOX

SOL_2
	btfsc	STARTED_REC,0
	call		STO_TIME		; store time delay
	bsf		STARTED_REC,0
; store solenoid number
	movlw	D'2'	
	movwi	FSR0++
	movlw	B'00000000'		; bit 5 cleared for drive solenoid
	movwi	FSR0++
; wait for switch open	
	call		DELAY			; 100ms debounce
S2_O
	movlw	D'1'
	call		DELAY_SET		; 10ms 
	btfsc	PORTA,4
	goto	S2_O
; Read stored value for COUNT and DUTY for solenoid 
	movlw	H'01'
	goto	STOX

SOL_3
	btfsc	STARTED_REC,0
	call		STO_TIME		; store time delay
	bsf		STARTED_REC,0
; store solenoid number
	movlw	D'3'	
	movwi	FSR0++
	movlw	B'00000000'		; bit 5 cleared for drive solenoid
	movwi	FSR0++
; wait for switch open
	call		DELAY			; 100ms debounce
S3_O
	movlw	D'1'
	call		DELAY_SET		; 10ms 
	btfsc	PORTC,5
	goto	S3_O
; Read stored value for COUNT and DUTY for solenoid 
	movlw	H'02'
	goto	STOX

SOL_4
	btfsc	STARTED_REC,0
	call		STO_TIME		; store time delay
	bsf		STARTED_REC,0
; store solenoid number
	movlw	D'4'	
	movwi	FSR0++
	movlw	B'00000000'		; bit 5 cleared for drive solenoid
	movwi	FSR0++
; wait for switch open
	call		DELAY			; 100ms debounce
S4_O
	movlw	D'1'
	call		DELAY_SET		; 10ms 
	btfsc	PORTC,4
	goto	S4_O
; Read stored value for COUNT and DUTY for solenoid
	movlw	H'03'
	goto	STOX

SOL_5
	btfsc	STARTED_REC,0
	call		STO_TIME		; store time delay
	bsf		STARTED_REC,0
; store solenoid number
	movlw	D'5'	
	movwi	FSR0++
	movlw	B'00000000'		; bit 5 cleared for drive solenoid
	movwi	FSR0++
; wait for switch open
	call		DELAY			; 100ms debounce
S5_O
	movlw	D'1'
	call		DELAY_SET		; 10ms 
	btfsc	PORTC,3
	goto	S5_O
; Read stored value for COUNT and DUTY for solenoid
	movlw	H'04'
	goto	STOX

SOL_6
	btfsc	STARTED_REC,0
	call		STO_TIME		; store time delay
	bsf		STARTED_REC,0
; store solenoid number
	movlw	D'6'	
	movwi	FSR0++
	movlw	B'00000000'		; bit 5 cleared for drive solenoid
	movwi	FSR0++
; wait for switch open
	call		DELAY			; 100ms debounce
S6_O
	movlw	D'1'
	call		DELAY_SET		; 10ms 
	btfsc	PORTC,6
	goto	S6_O
; Read stored value for COUNT and DUTY for solenoid
	movlw	H'05'
	goto	STOX

SOL_7
	btfsc	STARTED_REC,0
	call		STO_TIME		; store time delay
	bsf		STARTED_REC,0
; store solenoid number
	movlw	D'7'	
	movwi	FSR0++	
	movlw	B'00000000'		; bit 5 cleared for drive solenoid
	movwi	FSR0++
; wait for switch open
	call		DELAY			; 100ms debounce
S7_O
	movlw	D'1'
	call		DELAY_SET		; 10ms 
	btfsc	PORTC,7
	goto	S7_O
; Read stored value for COUNT and DUTY for solenoid
	movlw	H'06'
	goto	STOX

SOL_8
	btfsc	STARTED_REC,0
	call		STO_TIME		; store time delay
	bsf		STARTED_REC,0
; store solenoid number
	movlw	D'8'	
	movwi	FSR0++	
	movlw	B'00000000'		; bit 5 cleared for drive solenoid
	movwi	FSR0++
; wait for switch open
	call		DELAY			; 100ms debounce
S8_O
	movlw	D'1'
	call		DELAY_SET		; 10ms 
	btfsc	PORTB,7
	goto	S8_O
; Read stored value for COUNT and DUTY for solenoid
	movlw	H'07'
	goto	STOX

SOL_9
	btfsc	STARTED_REC,0
	call		STO_TIME		; store time delay
	bsf		STARTED_REC,0
; store solenoid number
	movlw	D'9'	
	movwi	FSR0++	
	movlw	B'00000000'		; bit 5 cleared for drive solenoid
	movwi	FSR0++
; wait for switch open
	call		DELAY			; 100ms debounce
S9_O
	movlw	D'1'
	call		DELAY_SET		; 10ms 
	btfsc	PORTC,2
	goto	S9_O
; Read stored value for COUNT and DUTY for solenoid
	movlw	H'08'
	goto	STOX

SOL_10
	btfsc	STARTED_REC,0
	call		STO_TIME		; store time delay
	bsf		STARTED_REC,0
; store solenoid number
	movlw	D'10'	
	movwi	FSR0++	
	movlw	B'00000000'		; bit 5 cleared for drive solenoid
	movwi	FSR0++
; wait for switch open
	call		DELAY			; 100ms debounce
S10_O
	movlw	D'1'
	call		DELAY_SET		; 10ms 
	btfsc	PORTB,6
	goto	S10_O
; Read stored value for COUNT and DUTY for solenoid
	movlw	H'09'
	goto	STOX

SOL_11
	btfsc	STARTED_REC,0
	call		STO_TIME		; store time delay
	bsf		STARTED_REC,0
; store solenoid number
	movlw	D'11'	
	movwi	FSR0++	
	movlw	B'00000000'		; bit 5 cleared for drive solenoid
	movwi	FSR0++
; wait for switch open
	call		DELAY			; 100ms debounce
S11_O
	movlw	D'1'
	call		DELAY_SET		; 10ms 
	btfsc	PORTB,5
	goto	S11_O
; Read stored value for COUNT and DUTY for solenoid
	movlw	H'0A'
	goto	STOX

SOL_12
	btfsc	STARTED_REC,0
	call		STO_TIME		; store time delay
	bsf		STARTED_REC,0
; store solenoid number
	movlw	D'12'	
	movwi	FSR0++
	movlw	B'00000000'		; bit 5 cleared for drive solenoid
	movwi	FSR0++
; wait for switch open
	call		DELAY			; 100ms debounce
S12_O
	movlw	D'1'
	call		DELAY_SET		; 10ms 
	btfsc	PORTB,4
	goto	S12_O
; Read stored value for COUNT and DUTY for solenoid
	movlw	H'0B'
;	goto	STOX

STOX
; ls flash address for each solenoid
	movwf	READADDLS	; ls address byte
; common fixed value
	movlw	H'10'			; ms address byte 
	movwf	READADDMS 
; read flash
	call		READ			; ms byte in TEMP_VAL
	movwf	TEMP			; ls byte in TEMP
; place in RAM

	movf		TEMP,w	
	movwi		FSR0++
	movf		TEMP_VAL,w
	movwi		FSR0++
; compare with H29AC
	movf		FSR0H,w
	sublw		H'29'
	movwf		TEMP
	movf		FSR0L,w
	sublw		H'AC'
	btfss		STATUS,C
	decf		TEMP,f
	btfsc		TEMP,7
	goto		END_RAM
	goto		RECORD1		

;............................................................................................................
; subroutine for storage of time delay between switches
STO_TIME
; when recording has started, store time delay first before switch values
; store in RAM

	movf		TIMER_LS,w
	movwi		FSR0++	
	movlw		B'11111111'		; for ms byte to set bit 5 of flash for pause flag
	movwi		FSR0++

	movf		TIMER_MS,w	
	movwi		FSR0++
	movlw		B'11111111'
	movwi		FSR0++

;clear counter
	clrf			TIMER_MS
	clrf			TIMER_LS	
	return
;........................................................................................................

DELAY_SET ; 10ms per count in w. must have value in w before call
; increases timer each 10ms
	movwf	VALUE_1
LP_SET
	clrf		TEN_MS		; 10ms flag
STAY_SET
	btfss	TEN_MS,0		; when set then 10ms is up
	goto	STAY_SET
	incfsz	TIMER_LS,f		; timer increased each 10ms
	goto	RED1x			; reduce value 1
	incf		TIMER_LS,f		; won't overrange in this time as starts from 0 and overranges after 10.9minutes
RED1x
	decfsz	VALUE_1,f		; reduce to zero
	goto	LP_SET
	return

; delay using 24MHz clock
DELAY		movlw	D'100'		; 100ms (1ms per value)
DELX		movwf	VALUE_3	; 1ms per value
DELT_1		movlw	D'16'		; set delay period  (D16)
			movwf	VALUE_1	; VALUE_1 = w
			movlw	D'130'		; set delay period value 2 (D130)
LP_1		movwf	VALUE_2	; VALUE_2 = w
LP_2		decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
			goto 	LP_2
			decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
			goto	LP_1
			decfsz	VALUE_3,f
			goto	DELT_1	
			return	

; shorter delay. Requires a value in w before calling DELY routine
DELY		movwf	VALUE_3	; 15.6us per value.
			movlw	D'30'		;  set delay period value 2 
LP_Y1		movwf	VALUE_2	; VALUE_2 = w
LP_Y2		decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
			goto 	LP_Y2
			decfsz	VALUE_3,f
			goto	LP_Y1
			return	

; subroutine to wait for A/D conversion 
ACQUIRE_AD

; wait >8us to charge input capacitance 
	movlw	D'50'
	movwf	STORE3
WAIT2C1
	decfsz	STORE3,f
	goto	WAIT2C1
	movlb	D'1'			; bank 1
	bsf		ADCON0,1	; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,1	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	movf	ADRESH,w	; value in w
	movlb	D'0'			; bank0
	return

;,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,
; write to calibration and natural flag memory
FLASH_SET	; write calibration values to flash
; Store in program memory 
	movlw	H'10'
	movwf	ADDRH			; upper byte flash address
	movlw	H'00'	;
	movwf	ADDRL			; lower byte 
; start of RAM
	movlw	H'20'
	movwf	FSR0H
	movlw 	H'50'
	movwf	FSR0L
	call		WRITE
	return
;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

; read data memory
READ; 'w' has read data
	bcf		 INTCON,GIE 	; Disable interrupts
	movlb	D'3'				;  bank 3 
	movf	READADDMS,w 
	movwf 	PMADRH		; ms Byte of Program Address to read
	movf	READADDLS,w
	movwf 	PMADRL 		; ls Byte of Program Address to read
	bcf		PMCON1,CFGS	; avoid configuration space
	bsf	 	PMCON1,RD 	; Read
	nop 					
	nop
; memory is read in second cycle PM read instruction
	movf	PMDATH,w 		; ms Byte of Program data 
	movwf	TEMP_VAL
	movf	PMDATL,w 		; ls Byte of Program data in 'w'
	movlb	D'0'				; bank 0
	bsf		 INTCON,GIE 	; enable interrupts
	return
;.............................................

; write to data memory
WRITE
	bcf		 INTCON,GIE 	; Disable interrupts so required sequences will execute properly
; load write latches
	movlb	 D'03'
	movf	ADDRH,W		; Load upper 6 bits of  address boundary			
	movwf	 PMADRH ;
	movf	 ADDRL,W 		; Load lower 8 bits of erase address boundary
	movwf	 PMADRL ;

; erase first
	bcf		 PMCON1,CFGS ; Not configuration space
	bsf		 PMCON1,FREE ; Specify an erase operation
	bsf		 PMCON1,WREN ; Enable writes
	movlw	H'55'			; Start of required write sequence:
	movwf	PMCON2 		; Write 55h
	movlw	H'AA' ;
	movwf	PMCON2 		; Write AAh
	bsf		PMCON1,WR 	; Set WR bit to begin erase
	nop					 	; NOP instructions are forced as processor writes all the program memory write latches simultaneously to program memory.
	nop	  
	bcf		PMCON1,WREN ; Disable writes

; load latches

	bcf		 PMCON1,CFGS	; Not configuration space
	bsf		 PMCON1,WREN	 ; Enable writes
	bsf		 PMCON1,LWLO 	; Only Load Write Latches

LOOPW
	moviw	FSR0++ 			; Load first data byte into upper
	movwf	PMDATL ;
	moviw	FSR0++ 	 		; Load second data byte into lower
	movwf	PMDATH ;

	movf	PMADRL,w		 ; Check  lower bits of address
	xorlw	0x1F			 ; Check if we're on the last of 32 addresses
	andlw	0x1F ;
	btfsc	STATUS,Z 		; Exit if last of 32 words,
	goto	START_WRITE ;


	movlw	 H'55'		 	; Start of required write sequence:
	movwf	 PMCON2 		; Write 55h
	movlw	 H'AA' ;
	movwf	 PMCON2		; Write AAh
	bsf		 PMCON1,WR 	; Set WR bit to begin write
	nop					 	; nop instructions are forced as processor loads program memory write latches
	nop ;
	incf		 PMADRL,f	 	; Still loading latches Increment address
	goto	 LOOPW

START_WRITE; write
	bcf		PMCON1,LWLO	; No more loading latches - Start Flash program
; memory write
	movlw	H'55'			; Start of required write sequence:
	movwf	PMCON2 		; Write 55h
	movlw	H'AA' ;
	movwf	PMCON2 		; Write AAh
	bsf		PMCON1,WR 	; Set WR bit to begin write
	nop						; NOP instructions are forced as processor writes all the program memory write latches simultaneously to program memory.
	nop	  
	bcf		PMCON1,WREN 	; Disable writes
	movlb	 D'0'
	bsf		 INTCON,GIE		; re-enable interrupts 
	return

;...................................................................................................................................................................
; Random number generator

;	Input:	32 bit initial integer seed in AARGB0, AARGB1, AARGB2, AARGB3

;	Use:	CALL	RAND32

;	Output:	32 bit random integer in AARGB0, AARGB1, AARGB2, AARGB3

;	Result:	AARG  <--  RAND32( AARG )

;	

;		min	max	mean
;	Timing:	487	487	487	clks



;	Linear congruential random number generator

;		X <- (a * X + c) mod m

;	The calculation is performed exactly, with multiplier a, increment c, and
;	modulus m, selected to achieve high ratings from standard spectral tests.

RAND32
		MOVF		RANDB0,W
		MOVWF		AARGB0
		MOVF		RANDB1,W
		MOVWF		AARGB1
		MOVF		RANDB2,W
		MOVWF		AARGB2
		MOVF		RANDB3,W
		MOVWF		AARGB3

		MOVLW		0x0D			; multiplier a = 1664525
		MOVWF		BARGB2
		MOVLW		0x66
		MOVWF		BARGB1
		MOVLW		0x19
		MOVWF		BARGB0

		CALL		FXM3224U

                INCF            AARGB6,F		; c = 1
                BTFSC           STATUS,Z
                INCF            AARGB5,F
		BTFSC		STATUS,Z
		INCF		AARGB4,F
		BTFSC		STATUS,Z
		INCF		AARGB3,F
		BTFSC           STATUS,Z
                INCF            AARGB2,F
		BTFSC		STATUS,Z
		INCF		AARGB1,F
		BTFSC		STATUS,Z
		INCF		AARGB0,F

		MOVF		AARGB3,W
		MOVWF		RANDB0			; m = 2**32
		MOVF		AARGB4,W
		MOVWF		RANDB1
		MOVF		AARGB5,W
		MOVWF		RANDB2
		MOVF		AARGB6,W
		MOVWF		RANDB3

		RETLW		0x00

;       32x24 Bit Unsigned Fixed Point Multiply 32x24 -> 56

;       Input:  32 bit unsigned fixed point multiplicand in AARGB0, AARGB1,
;               AARGB2, AARGB3

;               24 bit unsigned fixed point multiplier in BARGB0, BARGB1,
;               BARGB2

;       Use:    CALL    FXM3224U

;       Output: 56 bit unsigned fixed point product in AARGB0

;       Result: AARG  <--  AARG x BARG

;       Max Timing:     11+617+2 = 630 clks

;       Min Timing:     11+151 = 162 clks

;       PM: 11+139+1 = 151              DM: 15

FXM3224U
                CLRF    AARGB4          ; clear partial product
                CLRF    AARGB5
                CLRF    AARGB6
                MOVF   AARGB0,W
                MOVWF   TEMPB0
                MOVF   AARGB1,W
                MOVWF   TEMPB1
                MOVF   AARGB2,W
                MOVWF   TEMPB2
                MOVF   AARGB3,W
                MOVWF   TEMPB3

                CALL 	UMUL3224L

                RETLW           0x00

; UMUL3224L        macro

;       Max Timing:     2+15+6*25+24+2+7*26+25+2+7*27+26 = 617 clks

;       Min Timing:     2+7*6+5+1+7*6+5+1+7*6+5+6 = 151 clks

;       PM: 31+24+2+25+2+26+2+27 = 139            DM: 15

UMUL3224L       MOVLW   0x08
                MOVWF   LOOPCOUNT

LOOPUM3224A
                RRF     BARGB2,F
                BTFSC   STATUS,C
                GOTO    ALUM3224NAP
                DECFSZ  LOOPCOUNT,F
                GOTO    LOOPUM3224A

                MOVWF   LOOPCOUNT

LOOPUM3224B
                RRF     BARGB1,F
                BTFSC   STATUS,C
                GOTO    BLUM3224NAP
                DECFSZ  LOOPCOUNT,F
                GOTO    LOOPUM3224B

                MOVWF   LOOPCOUNT

LOOPUM3224C
                RRF     BARGB0,F
                BTFSC   STATUS,C
                GOTO    CLUM3224NAP
                DECFSZ  LOOPCOUNT,F
                GOTO    LOOPUM3224C

                CLRF    AARGB0
                CLRF    AARGB1
                CLRF    AARGB2
                CLRF    AARGB3
                RETLW   0x00
                
ALUM3224NAP     BCF     STATUS,C
                GOTO    ALUM3224NA
                
BLUM3224NAP     BCF     STATUS,C
                GOTO    BLUM3224NA
                
CLUM3224NAP     BCF     STATUS,C
                GOTO    CLUM3224NA

ALOOPUM3224
                RRF     BARGB2,F
                BTFSS   STATUS,C
                GOTO    ALUM3224NA
                MOVF    TEMPB3,W
                ADDWF   AARGB3,F
                MOVF            TEMPB2,W
                BTFSC           STATUS,C
                INCFSZ          TEMPB2,W
                ADDWF           AARGB2,F
                MOVF            TEMPB1,W
                BTFSC           STATUS,C
                INCFSZ          TEMPB1,W
                ADDWF           AARGB1,F
                MOVF            TEMPB0,W
                BTFSC           STATUS,C
                INCFSZ          TEMPB0,W
                ADDWF           AARGB0,F

ALUM3224NA
                RRF    AARGB0,F
                RRF    AARGB1,F
                RRF    AARGB2,F
                RRF             AARGB3,F
                RRF             AARGB4,F
                DECFSZ  LOOPCOUNT,F
                GOTO    ALOOPUM3224

                MOVLW   0x08
                MOVWF   LOOPCOUNT

BLOOPUM3224
                RRF     BARGB1,F
                BTFSS   STATUS,C
                GOTO    BLUM3224NA
                MOVF    TEMPB3,W
                ADDWF   AARGB3,F
                MOVF            TEMPB2,W
                BTFSC           STATUS,C
                INCFSZ          TEMPB2,W
                ADDWF           AARGB2,F
                MOVF            TEMPB1,W
                BTFSC           STATUS,C
                INCFSZ          TEMPB1,W
                ADDWF           AARGB1,F
                MOVF            TEMPB0,W
                BTFSC           STATUS,C
                INCFSZ          TEMPB0,W
                ADDWF           AARGB0,F

BLUM3224NA
                RRF    AARGB0,F
                RRF    AARGB1,F
                RRF    AARGB2,F
                RRF             AARGB3,F
                RRF             AARGB4,F
                RRF             AARGB5,F
                DECFSZ  LOOPCOUNT,F
                GOTO    BLOOPUM3224

                MOVLW   0x08
                MOVWF   LOOPCOUNT

CLOOPUM3224
                RRF     BARGB0,F
                BTFSS   STATUS,C
                GOTO    CLUM3224NA
                MOVF    TEMPB3,W
                ADDWF   AARGB3,F
                MOVF            TEMPB2,W
                BTFSC           STATUS,C
                INCFSZ          TEMPB2,W
                ADDWF           AARGB2,F
                MOVF            TEMPB1,W
                BTFSC           STATUS,C
                INCFSZ          TEMPB1,W
                ADDWF           AARGB1,F
                MOVF            TEMPB0,W
                BTFSC           STATUS,C
                INCFSZ          TEMPB0,W
                ADDWF           AARGB0,F

CLUM3224NA
                RRF    AARGB0,F
                RRF    AARGB1,F
                RRF    AARGB2,F
                RRF             AARGB3,F
                RRF             AARGB4,F
                RRF             AARGB5,F
                RRF             AARGB6,F
                DECFSZ  LOOPCOUNT,F
                GOTO    CLOOPUM3224
		RETURN

	end

